from abc import ABC, abstractmethod

class Tabuleiro:
    
    cores = ('branca', 'preta')
    
    # Método construtor do objeto
    def __init__(self):
        # Matriz de 8x8 com as posições das peças do tabuleiro
        self.__grade = [ [None] * 8 for _ in range(8) ]
        
        # Matriz utilizada para armazenar (temporariamente) a dominância das peças
        self.__dominancia = None
    
    # Coloca as peças em suas posições originais no tabuleiro
    def posicao_inicial(self):
        self.__init__()
        pecas = ['Pa2', 'Pb2', 'Pc2', 'Pd2', 'Pe2', 'Pf2', 'Pg2', 'Ph2', 
                 'Ta1', 'Cb1', 'Bc1', 'Rd1', 'We1', 'Bf1', 'Cg1', 'Th1']
        self.insere_pecas(pecas, 'branca')
        
        pecas = ['Pa7', 'Pb7', 'Pc7', 'Pd7', 'Pe7', 'Pf7', 'Pg7', 'Ph7', 
                 'Ta8', 'Cb8', 'Bc8', 'Rd8', 'We8', 'Bf8', 'Cg8', 'Th8']
        self.insere_pecas(pecas, 'preta')
    
    # Insere um peça no tabuleiro
    # Peca é dada uma string no formato como em 'Te5'
    # cor está em ['branca', 'preta']
    def insere_peca(self, peca, cor):
        nova_peca = PecaXadrez.from_string(peca, cor)
        linha = nova_peca.get_linha()
        coluna = nova_peca.get_coluna()
        if self.__grade[linha][coluna] is None:
            self.__grade[linha][coluna] = nova_peca
            self.__dominancia = None
        else:
            raise Exception(f"Casa já ocupada: {peca}")
        
    # pecas: coleção de peças no formato como em 'Te5'
    # cor está em ['branca' 'preta']
    def insere_pecas(self, pecas, cor):
        for peca in pecas:
            self.insere_peca(peca, cor)

    # Move uma peça no tabuleiro
    # A origem e o destino estão no formato como em 'f7'
    def move_peca(self, origem, destino):
        lin_orig, col_orig = Tabuleiro._converte_coordenada(origem)
        lin_dest, col_dest = Tabuleiro._converte_coordenada(destino)
        
        peca = self.__grade[lin_orig][col_orig]
        if peca is None:
            raise Exception(f"Não há peça em {origem}")
        
        p = self.__grade[lin_dest][col_dest]
        if p is not None and p.get_cor() == peca.get_cor():
            raise Exception(f"A posição de destino {destino} está ocupada")
        
        if not peca._pode_moverse(lin_dest, col_dest, self.__grade):
            raise Exception(f"Peça {str(peca)} que está em {origem} não pode ser movida para {destino}")
        
        peca._mova_para(lin_dest, col_dest)
        self.__grade[lin_dest][col_dest] = peca
        self.__grade[lin_orig][col_orig] = None
        self.__dominancia = None

    def roque(self, orig_p1, dest_p1, orig_p2, dest_p2):
        lin_orig, col_orig = Tabuleiro._converte_coordenada(orig_p1)
        lin_dest, col_dest = Tabuleiro._converte_coordenada(dest_p1)
        peca = self.__grade[lin_orig][col_orig]
        peca._mova_para(lin_dest, col_dest)
        self.__grade[lin_dest][col_dest] = peca
        self.__grade[lin_orig][col_orig] = None
        
        lin_orig, col_orig = Tabuleiro._converte_coordenada(orig_p2)
        lin_dest, col_dest = Tabuleiro._converte_coordenada(dest_p2)
        peca = self.__grade[lin_orig][col_orig]
        peca._mova_para(lin_dest, col_dest)
        self.__grade[lin_dest][col_dest] = peca
        self.__grade[lin_orig][col_orig] = None

    # Desenha o tabuleiro e as peças nele contidas
    def mostra(self):
        separacao = ['  '] + ['---']*8
        print(*separacao)
        for i in range(7, -1, -1):
            linha = self.__grade[i]
            print(i+1, end=' ')
            for j in range(8):
                peca = linha[j]
                txt = '   ' if peca is None else str(peca)+' '
                print('|'+txt, end='', sep='')
            print('|')
            print(*separacao)
        n = ord('a')
        itens = ['  ']
        for i in range(0, 8):
            itens.append(' ' + chr(n+i) + ' ')
        print(*itens)

    # Desenha o tabuleiro, as peças nele contidas e as dominâncias de cada casa
    def mostra_dominancia(self):
        if self.__dominancia is None:
            raise Exception("Dominância não foi calculada")
        
        separacao = ['  '] + ['---']*8
        print(*separacao)
        for i in range(7, -1, -1):
            linha = self.__grade[i]
            print(i+1, end=' ')
            for j in range(8):
                peca = linha[j]
                dom = ' ' if len(self.__dominancia[i][j]) == 0 else '*'
                txt = f' {dom} ' if peca is None else str(peca)+dom
                print('|'+txt, end='', sep='')
            print('|')
            print(*separacao)
        n = ord('a')
        itens = ['  ']
        for i in range(0, 8):
            itens.append(' ' + chr(n+i) + ' ')
        print(*itens)

    # Recalcula as dominâncias para uma das cores ['branca', 'preta']
    def calcula_dominancia(self, cor):
        self.__dominancia = [ [[] for _ in range(8)] for _ in range(8) ]
        for linha in self.__grade:
            for peca in linha:
                if peca is not None and peca.get_cor() == cor:
                    peca._marca_dominancia(self.__dominancia, self.__grade)

    # Verifica se o rei da cor está em cheque mate
    def esta_cheque_mate(self, cor):
        if self.__dominancia is None:
            self.calcula_dominancia(cor)
        rei = self.__get_rei(cor)
        if rei is None:
            raise Exception(f"Rei de cor {cor} não está no tabuleiro")
   
        # Se a casa onde o Rei está ou alguma das adjacentes não estiver dominada,
        # então o rei não está em cheque-mate
        lin_rei = rei.get_linha()
        col_rei = rei.get_coluna()
        for i in range(max(0,lin_rei-1), min(8,lin_rei+2)):
            for j in range(max(0,col_rei-1), min(8,col_rei+2)):
                if len(self.__dominancia[i][j]) == 0:
                    return False
        return True
    
    # ------ Métodos privados ou protegidos
    
    # Converte coordenada no padrão como 'e5' para coordenada da matriz correspondente ao tabuleiro
    @classmethod
    def _converte_coordenada(self, coordenada):
        if not coordenada[1].isdigit():
            raise Exception(f"Linha inválida: '{coordenada[1]}'")
        linha = int(coordenada[1])-1
        if linha not in range(0, 8):
            raise Exception(f"Linha inválida: '{coordenada[1]}'")
        
        if not coordenada[0].isalpha():
            raise Exception(f"Coluna inválida: '{coordenada[0]}'") 
        coluna = ord(coordenada[0]) - ord('a')
        if coluna not in range(0, 8):
            raise Exception(f"Coluna inválida: '{coordenada[0]}'")
        
        return (linha, coluna)
    
    # Retorna o rei da cor ou None caso ele não esteja no tabuleiro
    def __get_rei(self, cor):
        for linha in self.__grade:
            for peca in linha:
                if isinstance(peca, Rei) and peca.get_cor() == cor:
                    return peca
        return None

class PecaXadrez:
    
    # Método construtor de peça
    def __init__(self, cor='', linha=0, coluna=0):
        self._cor = cor
        self._linha = linha
        self._coluna = coluna
                
    # Método de classe que instantia uma peça a partir de sua descrição e cor
    # peca no formato como em "Te5"
    # cor em ['branca', 'preta']
    @classmethod
    def from_string(cls, peca, cor):
        if cor not in ['branca', 'preta']:
            raise Exception(f"Cor inválida: '{cor}'")
        
        if not peca[-1].isdigit():
            raise Exception(f"Linha inválida: '{peca[-1]}'")
        linha = int(peca[-1])-1
        if linha not in range(0, 8):
            raise Exception(f"Linha inválida: '{peca[-1]}'")
        
        if not peca[1].isalpha():
            raise Exception(f"Linha inválida: '{peca[1]}'") 
        coluna = ord(peca[1]) - ord('a')
        if coluna not in range(0, 8):
            raise Exception(f"Coluna inválida: '{peca[1]}'")
        
        match peca[0]:
            case 'W':
                return Rei(cor, linha, coluna)
            case 'R':
                return Rainha(cor, linha, coluna)
            case 'T':
                return Torre(cor, linha, coluna)
            case 'B':
                return Bispo(cor, linha, coluna)
            case 'C':
                return Cavalo(cor, linha, coluna)
            case 'P':
                return Peao(cor, linha, coluna)
            case _:
                raise Exception(f"Peça desconhecida: '{peca[0]}'")
  
    # Obtém a cor da peça
    def get_cor(self):
        return self._cor
  
    # Obtém a linha (escala 0..7) onde a peça está na matriz correspondente ao tabuleiro
    def get_linha(self):
        return self._linha
    
    # Obtém a coluna (escala 0..7) onde a peça está na matriz correspondente ao tabuleiro
    def get_coluna(self):
        return self._coluna
    
    # Move a peça para posicao lin, col, onde line e estão no intervalo [0..7]
    def _mova_para(self, lin, col):
        self._linha = lin
        self._coluna = col

    # Return True caso a peça seja Rei de cor diferente de cor
    def eh_rei_adversario(self, cor):
        return False

    # Verifica se a peça pode mover-se para uma determinada posição lin, col),
    # onde lin e col estão no intervalo [0..7]
    # grade: matriz que contém as peças
    @abstractmethod
    def _pode_moverse(self, lin, col, grade):
        pass
    
    # Verifica se há alguma peça no caminho da origem ao destino, com o acréscimos de delta
    def _caminho_livre(self, grade, lin_orig, col_orig, lin_dest, col_dest, lin_delta, col_delta):
        # A casa já está ocupada com peça da mesma cor
        peca = grade[lin_dest][col_dest]
        if peca is not None and peca.get_cor() == self.get_cor():
            return False
        
        # Verifica as casas entre a origem e o destino
        lin = lin_orig + lin_delta
        col = col_orig + col_delta
        while lin != lin_dest or col != col_dest:
            if grade[lin][col] is not None:
                return False
            lin += lin_delta
            col += col_delta
        return True
    
    # Marca dominância da peça para movimentos na mesma linha
    # grade: matriz que contém as peças
    def _marca_dominancia_linha(self, dominancia, grade):
        linha = self._linha
        
        # caminha para a direita na matriz
        for j in range(self._coluna+1, 8):
            dominancia[linha][j].append(self)
            if grade[linha][j] is not None and not grade[linha][j].eh_rei_adversario(self._cor):
                break
        # caminha para a esquerda na matriz
        for j in range(self._coluna-1, -1, -1):
            dominancia[linha][j].append(self)
            if grade[linha][j] is not None and not grade[linha][j].eh_rei_adversario(self._cor):
                break
            
    # Marca dominância da peça para movimentos na mesma coluna
    # grade: matriz que contém as peças
    def _marca_dominancia_coluna(self, dominancia, grade):
        coluna = self._coluna
        
        # caminha para baixo na matriz
        for i in range(self._linha+1, 8):
            dominancia[i][coluna].append(self)
            if grade[i][coluna] is not None and not grade[i][coluna].eh_rei_adversario(self._cor):
                break
        # caminha para cima na matriz
        for i in range(self._linha-1, -1, -1):
            dominancia[i][coluna].append(self)
            if grade[i][coluna] is not None and not grade[i][coluna].eh_rei_adversario(self._cor):
                break
            
    # Marca dominância da peça para movimentos no sentido da diagonal principal da matriz
    # grade: matriz que contém as peças
    def _marca_dominancia_diagonal_principal(self, dominancia, grade):
        linha = self._linha
        coluna = self._coluna
    
        # caminha para baixo e direita na matriz
        for i,j in zip(range(linha+1, 8), range(coluna+1, 8)):
            dominancia[i][j].append(self)
            if grade[i][j] is not None and not grade[i][j].eh_rei_adversario(self._cor):
                break
            
        # caminha para cima e esquerda na matriz
        for i,j in zip(range(linha-1, -1, -1), range(coluna-1, -1, -1)):
            dominancia[i][j].append(self)
            if grade[i][j] is not None and not grade[i][j].eh_rei_adversario(self._cor):
                break

    # Marca dominância da peça para movimentos no sentido da diagonal secundária da matriz
    # grade: matriz que contém as peças
    def _marca_dominancia_diagonal_secundaria(self, dominancia, grade):
        linha = self._linha
        coluna = self._coluna

        # caminha para baixo e esquerda na matriz
        for i,j in zip(range(linha+1, 8), range(coluna-1, -1, -1)):
            dominancia[i][j].append(self)
            if grade[i][j] is not None and not grade[i][j].eh_rei_adversario(self._cor):
                break
            
        # caminha para cima e direita na matriz
        for i,j in zip(range(linha-1, -1, -1), range(coluna+1, 8)):
            dominancia[i][j].append(self)
            if grade[i][j] is not None and not grade[i][j].eh_rei_adversario(self._cor):
                break

class Rei (PecaXadrez):
    
    # Método construtor de Rei
    def __init__(self, cor, linha, coluna):
        super().__init__(cor, linha, coluna)

    # Retorna uma representação em string "amigável" da peça que aparece que
    # usada em comandos como: print(peca) ou str(peca).
    def __str__(self):
        return 'W'+self._cor[0]
    
    # Marca dominância da peça na matriz de dominância
    # grade: matriz que contém as peças
    def _marca_dominancia(self, dominancia, grade):
        for i in range(max(0,self._linha-1), min(8,self._linha+2)):
            for j in range(max(0,self._coluna-1), min(8,self._coluna+2)):
                if i != self._linha or j != self._coluna:
                    dominancia[i][j].append(self)
        
    # Verifica se a peça pode mover-se para uma determinada posição lin, col),
    # onde lin e col estão no intervalo [0..7]
    # grade: matriz que contém as peças
    def _pode_moverse(self, lin, col, grade):
        # A casa já está ocupada com peça da mesma cor
        peca = grade[lin][col]
        if peca is not None and peca.get_cor() == self.get_cor():
            return False

        return abs(self._coluna - col) <= 1 and abs(self._linha - lin) <= 1

    # Return True caso a peça seja Rei de cor diferente de cor
    def eh_rei_adversario(self, cor):
        return self._cor != cor

class Rainha (PecaXadrez):
        
    # Método construtor de Rainha
    def __init__(self, cor, linha, coluna):
        super().__init__(cor, linha, coluna)

    # Retorna uma representação em string "amigável" da peça que aparece que
    # usada em comandos como: print(peca) ou str(peca).
    def __str__(self):
        return 'R'+self._cor[0]
    
    # Marca dominância da peça na matriz de dominância
    # grade: matriz que contém as peças
    def _marca_dominancia(self, dominancia, grade):
        self._marca_dominancia_linha(dominancia, grade)
        self._marca_dominancia_coluna(dominancia, grade)
        self._marca_dominancia_diagonal_principal(dominancia, grade)
        self._marca_dominancia_diagonal_secundaria(dominancia, grade)
        
    # Verifica se a peça pode mover-se para uma determinada posição lin, col),
    # onde lin e col estão no intervalo [0..7]
    # grade: matriz que contém as peças
    def _pode_moverse(self, lin, col, grade):
        dlin = 1 if self._linha < lin else -1
        dcol = 1 if self._coluna < col else -1
        if self._linha == lin:
            dlin = 0
        elif self._coluna == col:
            dcol = 0
            
        return (self._coluna == col or self._linha == lin or \
                   abs(self._coluna - col) == abs(self._linha - lin)) \
               and (self._coluna != col or self._linha != lin) \
               and self._caminho_livre(grade, self._linha, self._coluna, lin, col, dlin, dcol)
        
class Torre (PecaXadrez):
        
    # Método construtor de Torre
    def __init__(self, cor, linha, coluna):
        super().__init__(cor, linha, coluna)

    # Retorna uma representação em string "amigável" da peça que aparece que
    # usada em comandos como: print(peca) ou str(peca).
    def __str__(self):
        return 'T'+self._cor[0]
    
    # Marca dominância da peça na matriz de dominância
    # grade: matriz que contém as peças    
    def _marca_dominancia(self, dominancia, grade):
        self._marca_dominancia_linha(dominancia, grade)
        self._marca_dominancia_coluna(dominancia, grade)
        
    # Verifica se a peça pode mover-se para uma determinada posição lin, col),
    # onde lin e col estão no intervalo [0..7]
    # grade: matriz que contém as peças
    def _pode_moverse(self, lin, col, grade):
        if self._linha == lin:
            dlin = 0
            dcol = 1 if self._coluna < col else -1
        else:
            dcol = 0
            dlin = 1 if self._linha < lin else -1
        return (self._linha == lin or self._coluna == col) \
               and (self._coluna != col or self._linha != lin) \
               and self._caminho_livre(grade, self._linha, self._coluna, lin, col, dlin, dcol)

class Bispo (PecaXadrez):
        
    # Método construtor de Bispo
    def __init__(self, cor, linha, coluna):
        super().__init__(cor, linha, coluna)

    # Retorna uma representação em string "amigável" da peça que aparece que
    # usada em comandos como: print(peca) ou str(peca).
    def __str__(self):
        return 'B'+self._cor[0]
    
    # Marca dominância da peça na matriz de dominância
    # grade: matriz que contém as peças    
    def _marca_dominancia(self, dominancia, grade):
        self._marca_dominancia_diagonal_principal(dominancia, grade)
        self._marca_dominancia_diagonal_secundaria(dominancia, grade)
        
    # Verifica se a peça pode mover-se para uma determinada posição lin, col),
    # onde lin e col estão no intervalo [0..7]
    # grade: matriz que contém as peças
    def _pode_moverse(self, lin, col, grade):
        dlin = 1 if self._linha < lin else -1
        dcol = 1 if self._coluna < col else -1
        return abs(self._coluna - col) == abs(self._linha - lin) \
               and (self._coluna != col or self._linha != lin) \
               and self._caminho_livre(grade, self._linha, self._coluna, lin, col, dlin, dcol)

class Cavalo (PecaXadrez):
        
    # Método construtor de Cavalo
    def __init__(self, cor, linha, coluna):
        super().__init__(cor, linha, coluna)

    # Retorna uma representação em string "amigável" da peça que aparece que
    # usada em comandos como: print(peca) ou str(peca).
    def __str__(self):
        return 'C'+self._cor[0]

    # Marca dominância da peça na matriz de dominância
    # grade: matriz que contém as peças    
    def _marca_dominancia(self, dominancia, grade):
        for di in (1, -1, 2, -2):
            i = self._linha + di
            for dj in (1, -1, 2, -2):
                j = self._coluna + dj
                if abs(di)+abs(dj) == 3 and i in range(0,8) and j in range(0,8):
                    dominancia[i][j].append(self)
        
    # Verifica se a peça pode mover-se para uma determinada posição lin, col),
    # onde lin e col estão no intervalo [0..7]
    # grade: matriz que contém as peças
    def _pode_moverse(self, lin, col, grade):
        # A casa já está ocupada com peça da mesma cor
        peca = grade[lin][col]
        if peca is not None and peca.get_cor() == self.get_cor():
            return False
        
        return abs(self._linha-lin) + abs(self._coluna-col) == 3 \
               and self._linha != lin and self._coluna != col

# Método construtor de Peão
class Peao (PecaXadrez):
    
    def __init__(self, cor, linha, coluna):
        super().__init__(cor, linha, coluna)

    # Retorna uma representação em string "amigável" da peça que aparece que
    # usada em comandos como: print(peca) ou str(peca).
    def __str__(self):
        return 'P'+self._cor[0]
    
    # Marca dominância da peça na matriz de dominância
    # grade: matriz que contém as peças    
    def _marca_dominancia(self, dominancia, grade):
        if self._cor == 'branca' and self._linha < 7:
            dominancia[self._linha+1][self._coluna-1].append(self)
            dominancia[self._linha+1][self._coluna+1].append(self)
        elif self._cor == 'preta' and self._linha >= 1:
            dominancia[self._linha-1][self._coluna-1].append(self)
            dominancia[self._linha-1][self._coluna+1].append(self)
        
    # Verifica se a peça pode mover-se para uma determinada posição lin, col),
    # onde lin e col estão no intervalo [0..7]
    # grade: matriz que contém as peças
    def _pode_moverse(self, lin, col, grade):
        if self._coluna == col:
            if self._cor == 'branca':
                return lin-self._linha in range(1,3) \
                       and self._caminho_livre(grade, self._linha, self._coluna, lin, col, 1, 0) \
                       and grade[lin][col] is None
            else:
                return self._linha-lin in range(1,3) \
                       and self._caminho_livre(grade, self._linha, self._coluna, lin, col, -1, 0) \
                       and grade[lin][col] is None
        else:
            peca = grade[lin][col]
            if peca is not None and peca.get_cor() == self.get_cor():
                # A casa já está ocupada com peça da mesma cor
                return False
            if self._cor == 'branca':
                return lin == self._linha+1 and abs(self._coluna-col) == 1
            else:
                return lin == self._linha-1 and abs(self._coluna-col) == 1